"use client";
import React, { useState, useEffect, useMemo } from "react";
import Select from "react-select";

const InputField = ({ label, value, onChange, placeholder, error }) => (
  <div className="mb-4">
    <label className="block text-base font-bold text-gray-700 ml-2">
      {label}
    </label>
    <input
      className={`w-80 px-3 py-2 border rounded-md focus:outline-none text-black focus:border-${
        error ? "red-500" : "blue-500"
      }`}
      type="text"
      value={value}
      onChange={onChange}
      placeholder={placeholder}
    />
    {error && <p className="text-red-500 text-sm mt-1 ml-1">{error}</p>}
  </div>
);

const UsuarioForm = ({ accion, usuario, listaPermisos, onClose, onSubmit }) => {
  const titles = {
    agregar: "Agregar Nuevo Usuario",
    editar: "Editar Usuario",
  };

  const [nombre, setNombre] = useState("");
  const [rut, setRut] = useState("");
  const [telefono, setTelefono] = useState("");
  const [correo, setCorreo] = useState("");
  const [nombreError, setNombreError] = useState("");
  const [rutError, setRutError] = useState("");
  const [telefonoError, setTelefonoError] = useState("");
  const [correoError, setCorreoError] = useState("");
  const [permiso, setPermiso] = useState(null);
  const [datosCargados, setDatosCargados] = useState(false);
  const [formIsValid, setFormIsValid] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");

  const validateField = (value, setError, regex, errorMessage) => {
    if (!value) {
      setError(`${errorMessage} es requerido`);
    } else if (!regex.test(value)) {
      setError(`Formato inválido para ${errorMessage}`);
    } else {
      setError("");
    }
  };

  useEffect(() => {
    const isValid =
      nombreError === "" &&
      rutError === "" &&
      telefonoError === "" &&
      correoError === "" &&
      permiso !== null;

    setFormIsValid(isValid);
  }, [nombreError, rutError, telefonoError, correoError, permiso]);

  const handleNombreChange = (e) => {
    const value = e.target.value;
    const filteredValue = value.replace(/[^A-Za-zÁÉÍÓÚáéíóúñÑ\s]/g, "");
    const limitedValue = filteredValue.slice(0, 30);
    setNombre(limitedValue);
    validateField(
      limitedValue,
      setNombreError,
      /^[A-Za-zÁÉÍÓÚáéíóúñÑ\s]{1,30}$/,
      "Nombre"
    );
  };

  const handleRutChange = (e) => {
    const value = e.target.value;
    const numericValue = value.replace(/[^\dKk]/g, "");
    const formattedRut = numericValue.replace(
      /^(\d{1,2})(\d{3})(\d{3})(\w{1,2})$/,
      "$1.$2.$3-$4"
    );
    const limitedRut = formattedRut.slice(0, 12);
    setRut(limitedRut);
    validateField(
      limitedRut,
      setRutError,
      /^\d{1,2}\.\d{3}\.\d{3}-[\dKk]$/,
      "RUT"
    );
  };

  const handleTelefonoChange = (e) => {
    const value = e.target.value;
    const numericValue = value.replace(/[^\d]/g, "");
    const limitedValue = numericValue.slice(0, 9);
    setTelefono(limitedValue);
    validateField(limitedValue, setTelefonoError, /^\d{9}$/, "Teléfono");
  };

  const handleCorreoChange = (e) => {
    const value = e.target.value;
    setCorreo(value);
    validateField(
      value,
      setCorreoError,
      /^[A-Za-zÁÉÍÓÚáéíóúñÑ\d._%+-]+@[A-Za-zÁÉÍÓÚáéíóúñÑ\d.-]+\.[A-Za-zÁÉÍÓÚáéíóúñÑ]{2,}$/,
      "Correo Electrónico"
    );
  };

  const opcionesPermisos = useMemo(() => {
    return listaPermisos.map((permiso) => ({
      value: permiso.value,
      label: permiso.label,
      id: permiso.value,
      nombre: permiso.label,
    }));
  }, [listaPermisos]);

  const resetForm = () => {
    setNombre("");
    setRut("");
    setTelefono("");
    setCorreo("");
    setNombreError("");
    setRutError("");
    setTelefonoError("");
    setCorreoError("");
    setPermiso(null);
  };

  const handleSubmit = (e) => {
    e.preventDefault();

    if (formIsValid) {
      const permisoCambiado =
        permiso &&
        usuario?.permisos_usuarios[0]?.id_permisos_usr !== permiso.value;

      const usuarioData = {
        usuario_id: accion === "agregar" ? null : usuario?.usuario_id,
        nombre_usuario: nombre,
        rut_usuario: rut,
        telefono_usuario: telefono,
        correo_usuario: correo,
        permisos_usuarios: permisoCambiado
          ? permiso
            ? [
                {
                  id_permisos_usr: permiso.value,
                  nombre_permisos: permiso.label,
                },
              ]
            : []
          : usuario?.permisos_usuarios || [],
      };

      onSubmit(accion, usuarioData);
      onClose();
    } else {
      // Mover la lógica de mostrar el mensaje después de intentar enviar el formulario
      setErrorMessage("Llene correctamente todos los campos");
    }
  };

  const handleClose = () => {
    resetForm();
    onClose();
  };

  const customStyles = {
    // Estilo para el menú del Select
    menuPortal: (base) => ({
      ...base,
      zIndex: 100,
      color: "#3a4151",
      hidden: "auto",
      cursor: "pointer",
    }),
  };

  useEffect(() => {
    if (accion === "agregar" && usuario) {
      resetForm();
      setDatosCargados(true);
    } else if (accion === "editar" && usuario && !datosCargados) {
      setNombre(usuario?.nombre_usuario || "");
      setRut(usuario?.rut_usuario || "");
      setTelefono(usuario?.telefono_usuario || "");
      setCorreo(usuario?.correo_usuario || "");
      if (usuario?.permisos_usuarios?.length > 0) {
        const permisoSeleccionado = usuario.permisos_usuarios[0];
        setPermiso({
          value: permisoSeleccionado.id_permisos_usr,
          label: permisoSeleccionado.nombre_permisos,
        });
        setDatosCargados(true);
      } else {
        setPermiso(null);
      }
    }
  }, [accion, usuario, listaPermisos, datosCargados]);

  return (
    <form className="overflow-auto" onSubmit={handleSubmit}>
      <div className="text-black w-3/4 border-b border-black mb-4">
        <h2 className="text-black text-xl font-bold mb-[1px] ml-1">
          {titles[accion]}
        </h2>
      </div>
      <div className="">
        <InputField
          label="Nombre del Usuario:"
          value={nombre}
          onChange={handleNombreChange}
          placeholder="Ingrese el nombre"
          error={nombreError}
        />
        <InputField
          label="Rut del Usuario:"
          value={rut}
          onChange={handleRutChange}
          placeholder="Ingrese el RUT"
          error={rutError}
        />
        <InputField
          label="Teléfono del Usuario:"
          value={telefono}
          onChange={handleTelefonoChange}
          placeholder="Ingrese Teléfono"
          error={telefonoError}
        />
        <InputField
          label="Correo Electrónico del Usuario:"
          value={correo}
          onChange={handleCorreoChange}
          placeholder="Ingrese Correo Electrónico"
          error={correoError}
        />
        <label className="block text-base font-bold text-gray-600 ml-2 mb-1">
          Permisos Usuario
        </label>
        <Select
          options={opcionesPermisos}
          placeholder="Seleccione Permiso"
          className="text-black"
          value={permiso}
          onChange={(selected) => setPermiso(selected)}
          styles={customStyles}
          menuPortalTarget={document.body}
        />
      </div>
      <div className="mt-4 text-right">
        <button
          type="submit"
          className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg mr-2"
        >
          {accion === "agregar" ? "Agregar" : "Guardar"}
        </button>
        <button
          className="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded-lg"
          onClick={handleClose}
        >
          Cerrar
        </button>
        {errorMessage && (
          <p className="text-red-500 text-sm mt-2">{errorMessage}</p>
        )}
      </div>
    </form>
  );
};

export default UsuarioForm;
