import React, { useState } from "react";
import ModalTerminos from "./modalterminos";
import axios from "axios";
import { API_URL } from "../../const";

const Input = ({ label, type, placeholder, value, onChange, error }) => {
  return (
    <div className="mb-2 ">
      <label
        className="block text-gray-700 text-base font-bold mb-1 ml-3 text-left"
        htmlFor={label}
      >
        {label}
      </label>
      <input
        className={`shadow appearance-none border w-full mb-1 rounded  py-3 px-3 text-gray-700 focus:border-blue-400 leading-tight focus:outline-none focus:shadow-outline ${
          error ? "border-red-500" : ""
        }`}
        id={label}
        type={type}
        placeholder={placeholder}
        value={value}
        onChange={onChange}
      />
      {error && (
        <p className="text-red-500 text-xs italic ml-4 text-left w-[70%]">
          {error}
        </p>
      )}
    </div>
  );
};

const FormularioIngreso = ({ token, ip, onFinish }) => {
  const [nombre, setNombre] = useState("");
  const [rut, setRut] = useState("");
  const [password, setPassword] = useState("");
  const [correo, setCorreo] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [passwordsMatch, setPasswordsMatch] = useState(true);
  const [validationErrors, setValidationErrors] = useState({
    nombre: "",
    correo: "",
    rut: "",
    password: "",
    confirmPassword: "",
    general: "",
  });

  const [valid, setValid] = useState(false);

  const [modalOpen, setModalOpen] = useState(false);

  const openModal = () => {
    setModalOpen(true);
  };

  const closeModal = () => {
    setModalOpen(false);
  };

  const onValidPress = () => {
    setValid(!valid);
  };

  const onAcceptTerms = () => {
    const postFormulario = async () => {
      try {
        const response = await axios.post(
          API_URL + "/ingreso-empresaformulario-completar",
          {
            token_formulario: token,
            rut_usuario: rut,
            password: password,
            nombre_ejecutivo: nombre,
            correo_ejecutivo: correo,
            ip_ejecutivo: ip,
            pc_ejecutivo: "",
          }
        );
        // Procesa los datos de respuesta según sea necesario
        onFinish();
      } catch (error) {
        // Manejo de errores
        console.error("Error posting data:", error);
      }
    };
    postFormulario();
    console.log("Formulario terminado");
  };

  const formatRut = (inputRut, maxLength) => {
    let rutFormatted = inputRut.slice(0, maxLength);
    rutFormatted = rutFormatted.replace(/[^\dkK]/g, "");
    if (rutFormatted.length > 1) {
      rutFormatted = rutFormatted.replace(
        /^(\d{1,2})(\d{3})(\d{3})/,
        "$1.$2.$3"
      );
      rutFormatted = rutFormatted.replace(/^(\d{2}\.\d{3}\.\d{3})/, "$1-");
    }
    return rutFormatted;
  };

  const handleRutChange = (e) => {
    const inputRut = e.target.value;
    const formattedRut = formatRut(inputRut, 12); // Cambia el segundo parámetro según el máximo de caracteres deseados
    setRut(formattedRut);
  };

  const handleNombreChange = (e) => {
    const inputNombre = e.target.value;
    const onlyLettersAndAccents = /^[A-Za-zñÑáéíóúÁÉÍÓÚ\s]+$/;

    if (onlyLettersAndAccents.test(inputNombre) || inputNombre === "") {
      setNombre(inputNombre);
    }
  };

  const handleCorreoChange = (e) => {
    const inputCorreo = e.target.value;

    setCorreo(inputCorreo);
  };

  const handlePasswordChange = (e) => {
    const newPassword = e.target.value;

    setValidationErrors({
      ...validationErrors,
      password:
        newPassword.length >= 8 &&
        /[A-Z]/.test(newPassword) &&
        /\d/.test(newPassword)
          ? ""
          : "La contraseña debe tener al menos 8 caracteres, una mayúscula y un número.",
    });

    setPassword(newPassword);
  };

  const handleConfirmPasswordChange = (e) => {
    const confirmPasswordValue = e.target.value;

    setConfirmPassword(confirmPasswordValue);

    setValidationErrors((prevErrors) => ({
      ...prevErrors,
      confirmPassword:
        confirmPasswordValue === password
          ? ""
          : "Las contraseñas no coinciden.",
    }));

    // Actualiza el estado de passwordsMatch
    setPasswordsMatch(confirmPasswordValue === password);
  };

  const handleClose = () => {
    resetForm();
  };

  const resetForm = () => {
    setNombre("");
    setRut("");
    setCorreo("");
    setPassword("");
    setConfirmPassword("");
    setPasswordsMatch(true);
    setValidationErrors({
      nombre: "",
      rut: "",
      password: "",
      confirmPassword: "",
      general: "",
    });
  };

  const handleSubmit = (e) => {
    e.preventDefault();

    console.log(rut.length);
    console.log(isNaN(rut.replace(/[^\dkK]/g, "")));

    setValidationErrors({
      nombre: nombre ? "" : "El nombre es obligatorio.",
      rut: rut.length === 12 ? "" : "El RUT debe tener 12 caracteres.",
      password:
        password.length >= 8 && /[A-Z]/.test(password) && /\d/.test(password)
          ? ""
          : "La contraseña debe tener al menos 8 caracteres, una mayúscula y un número.",
      confirmPassword: passwordsMatch ? "" : "Las contraseñas no coinciden.",
      general:
        !nombre || !rut || !password || !confirmPassword
          ? "Todos los campos son obligatorios."
          : "",
      correo: correo ? "" : "El correo es obligatorio.",
    });

    if (Object.values(validationErrors).some((error) => error !== "")) {
      return;
    }

    openModal();
    console.log("Nombre:", nombre);
    console.log("RUT:", rut);
    console.log("Contraseña:", password);
    setPasswordsMatch(true);
  };

  return (
    <div>
      <div className="bg-white p-5 rounded-lg relative modal-content text-black shadow-md">
        <form className="w-full">
          <Input
            label="Nombre"
            type="text"
            placeholder="Ingrese su nombre"
            value={nombre}
            onChange={(e) => {
              handleNombreChange(e);
              setValidationErrors({ ...validationErrors, nombre: "" });
            }}
            error={validationErrors.nombre}
          />

          <Input
            label="Correo"
            type="text"
            placeholder="Ingrese su correo"
            value={correo}
            onChange={(e) => {
              handleCorreoChange(e);
              setValidationErrors({ ...validationErrors, correo: "" });
            }}
            error={validationErrors.correo}
          />

          <Input
            label="RUT"
            type="text"
            placeholder="Ingrese su RUT"
            value={rut}
            onChange={(e) => {
              handleRutChange(e);
              setValidationErrors({ ...validationErrors, rut: "" });
            }}
            error={validationErrors.rut}
          />
          <Input
            label="Contraseña"
            type="password"
            placeholder="Ingrese su contraseña"
            value={password}
            onChange={handlePasswordChange}
            error={validationErrors.password}
          />
          <Input
            label="Confirmar Contraseña"
            type="password"
            placeholder="Confirme su contraseña"
            value={confirmPassword}
            onChange={(e) => {
              handleConfirmPasswordChange(e);
              setValidationErrors({ ...validationErrors, confirmPassword: "" });
            }}
            error={validationErrors.confirmPassword}
          />
          {validationErrors.general && (
            <p className="text-red-500 text-xs italic ml-4 text-left w-[70%]">
              {validationErrors.general}
            </p>
          )}
          <div className="flex text-center gap-4 justify-center mt-6">
            <button
              className="bg-red-500 hover:bg-red-700 text-lg text-white font-bold py-3 px-4 rounded focus:outline-none focus:shadow-outline"
              type="button"
              onClick={handleClose}
            >
              Cerrar
            </button>
            <button
              className="bg-blue-500 hover:bg-blue-700 text-lg text-white font-bold py-3 px-4 rounded focus:outline-none focus:shadow-outline"
              onClick={handleSubmit}
            >
              Enviar
            </button>
          </div>
          <ModalTerminos
            isOpen={modalOpen}
            onClose={closeModal}
            valid={valid}
            onCheck={onValidPress}
            onAccept={onAcceptTerms}
          />
        </form>
      </div>
    </div>
  );
};

export default FormularioIngreso;
